/******************************************************/
/* Examples Program For "ET-BASE dsPIC30F2010"  Board */
/* Hardware    : ET-BASE dsPIC30F2010                 */
/* Target MCU  : dsPIC30F2010                         */
/*       	   : X-TAL : 7.3728 MHz                   */
/*             : Run 117.9648MHz                      */
/*             : Selec OSC Mode = XT w/PLL 16x        */
/* Compiler    : MPLAB v8.40 + C30 v3.20B      	      */
/* Last Update : 12/October/2009                      */
/******************************************************/
/* Function    : Example Use Capture 16 Bit Mode      */
/*             : Used RD0 Capture Measure Pulse 1KHz  */
/******************************************************/
/* Used Timer2 Count Cycle of Capture Pulse           */
/* Used Capture1(RD0) Measure Pulse 1 KHz             */
/* Minimum Period(65535 Cycle)=2.2mS (450 Hz)         */
/* Maximum Period(1000 Cycle)=33.9uS(29.4912KHz)      */
/* Display Result to Serial Port UART1(9600BPS)       */
/******************************************************/

#include <p30f2010.h>                                		// dsPIC30F2010 MPU Register
#include <stdio.h>											// Used "sprintf" Function
#include "uart.h"											// Used UART Config
#include "incap.h"											// Used Input Capture Library Function
#include "timer.h"											// Used Timer Library Function

/* Setup Configuration For ET-BASE dsPIC30F2010 */
_FOSC(CSW_FSCM_OFF & XT_PLL16);								// Disable Clock Switching,Enable Fail-Salf Clock
                                                            // Clock Source = Primary XT + (PLL x 16)
_FWDT(WDT_OFF);												// Disable Watchdog 
_FBORPOR(PBOR_OFF & PWRT_64 & MCLR_EN);					    // Disable Brown-Out ,Power ON = 64mS,Enable MCLR
_FGS(CODE_PROT_OFF);										// Code Protect OFF
/* End Configuration For ET-BASE dsPIC30F2010 */

unsigned int Capture_Interrupt_Count;						// Capture Edge Count
unsigned int Capture_Status;								// Capture Status
unsigned int Capture_Value1;								// 1st Capture Value Save
unsigned int Capture_Value2;		     					// 2nd Capture Value Save
char uart_buf[40];											// "sprintf" Buffer

/* pototype  section */
void init_uart(void);										// Initial UART1 Function
void init_capture(void);									// Initial Capture Function

/******************************/
/* Capture1 Interrupt Service */
/******************************/
void __attribute__((__interrupt__, no_auto_psv)) _IC1Interrupt(void)
{  
  Capture_Interrupt_Count++;								// Count Edge Detect
  if (Capture_Interrupt_Count==1) 							// First Edge Detect
  {    
    ReadCapture1(&Capture_Value1); 							// Save 1st Edge Capture Value        
	Capture_Status = 1;										// Set Capture Complete Status   
  }    
      
  else if (Capture_Interrupt_Count==2)						// Second Edge Detect
  { 
    ReadCapture1(&Capture_Value2); 							// Save 2nd Edge Capture Value        
	Capture_Status = 1;										// Set Capture Complete Status   
  }
  
  IFS0bits.IC1IF = 0;										// Reset Capture1 Interrupt Flag  
}


// Main Program Start Here //
int main(void)
{  
  unsigned int Period;										// Period Value
   
  init_uart();												// Initial UART = 9600,N,8,1
      
  sprintf(uart_buf,"ET-BASE dsPIC30F2010\n\r");				// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete
  
  sprintf(uart_buf,"Demo Capture1 Measure Pulse\n\n\r");	// Print Message String 
  putsUART1((unsigned int *)uart_buf);                      // Print uart_buff to UART1
  while(BusyUART1());										// Wait putsUART1 Complete  
  
  /* Start Used Capture1 Measure Pulse on RD0 Pin */
  while(1)													// Loop Continue
  {                 
     init_capture();										// Initial Capture1 Function
     Capture_Status = 0;									// Reset Status  
     Capture_Interrupt_Count = 0;							// Reset Edge Count

     while(Capture_Status == 0);							// Wait Edge Capture Complete
     Capture_Status = 0;									// Reset Status
	 while(Capture_Status == 0);							// Wait Edge Capture Complete	 
     CloseCapture1();										// Stop Capture1 Function	
      
     if (Capture_Value2 > Capture_Value1) 
     {
       Period = Capture_Value2 - Capture_Value1;
     }
     else
     {
       Period = (65535-Capture_Value1)+(Capture_Value2+1);       
     }
    
     // Display Capture Value in Cycle (1/29.4912MHz)
     // Frequency 1 KHz = 29491.2 Cycle
     sprintf(uart_buf,"Period = %5u Cycle\r",Period);		// Print Message String 
     putsUART1((unsigned int *)uart_buf);                   // Print uart_buff to UART1
     while(BusyUART1());									// Wait putsUART1 Complete
  } 
}	  

/*********************************/
/* Initial UART for dsPIC30F2010 */
/* 9600,N,8,1 / 117.9648MHz Fosc */
/*********************************/
void init_uart(void)
{		  
  CloseUART1();												// Disable UART1 Before New Config

  // Config UART1 Interrupt Control
  ConfigIntUART1(UART_RX_INT_DIS &							// Disable RX Interrupt
    		     UART_RX_INT_PR2 &							// RX Interrupt Priority = 2
    		     UART_TX_INT_DIS &							// Disable TX Interrupt
    		     UART_TX_INT_PR3);							// TX Interrupt Priority = 3  

  // Open UART1 = Mode,Status,Baudrate              
  OpenUART1(UART_EN	&										// Enable UART(UART Mode)
            UART_IDLE_STOP &								// Disable UART in IDLE Mode 
 			UART_ALTRX_ALTTX & 								// Select U1TX=RC13,U1RX=RC14
            UART_DIS_WAKE &									// Disable Wake-Up
			UART_DIS_LOOPBACK &								// Disable Loop Back
			UART_DIS_ABAUD &								// Disable Auto Baudrate
  			UART_NO_PAR_8BIT &								// UART = 8 Bit, No Parity
 			UART_1STOPBIT,									// UART = 1 Stop Bit

	  		// Config UART1 Status
 			UART_INT_TX & 									// Select Interrupt After TX Complete
	 		UART_TX_PIN_NORMAL &							// Normal U1TX Mode
 			UART_TX_ENABLE &								// Enable U1TX
 	 		UART_INT_RX_BUF_FUL &							// Flasg Set After RX Complete 
  			UART_ADR_DETECT_DIS &              				// Disable Check Address 
			UART_RX_OVERRUN_CLEAR,							// Clear Overrun Flag

  			// ET-BASE dsPIC30F2010 Hardware Board
  			// XTAL = 7.3728MHz
  			// Fosc = 7.3728 MHz x 16 = 117.9648 MHz
  			// Fcy(UART) = Fosc / 4 
  			//           = 117.9648 / 4 = 29.4912 MHz
  			// U1BRG = [Fcy/(16xBaud)]-1
  			//       = [29.4912 MHz / (16x9600)] - 1
  			//       = 191 = BFH			
  			191);											// ET-BASE dsPIC30F2010 UART Baudrate = 9600 BPS
}

/**********************************/
/* Initial Timer for dsPIC30F2010 */
/**********************************/
void init_capture(void)
{		  
  CloseCapture1();											// Disable Capture1 Befor New Config
  CloseTimer2();											// Disable Timer2 Befor New Config

  // Config Capture1 Interrupt Control
  ConfigIntCapture1(IC_INT_ON &								// Enable Capture Interrupt
					IC_INT_PRIOR_6);						// Capture Interrupt Priority = 6
                    
  // Config Timer2 Interrupt Control
  ConfigIntTimer2(T2_INT_OFF &								// Disable Timer2 Interrupt
                  T2_INT_PRIOR_7);							// Timer2 Interrupt Priority = 7
  
  //***************************************************/
  // ET-BASE dsPIC30F2010 Hardware Board              */
  // XTAL = 7.3728MHz								  */
  // Fosc = 7.3728 MHz x 16 = 117.9648 MHz			  */	
  // Fcy  = Fosc / 4 								  */
  //      = 117.9648 / 4 = 29.4912 MHz				  */
  // Tcy  = 1 / 29.4912 MHz							  */	
  //      = 33.90842 nS								  */	
  //***************************************************/
    
  OpenTimer2(T2_ON &										// ON Timer2
  			 T2_IDLE_STOP &									// Disable Timer2 in IDLE Mode
             T2_32BIT_MODE_OFF &							// Timer2 = 16 Bit Timer
  			 T2_GATE_OFF &									// Disable Timer2 Gate Control
             T2_PS_1_1 &									// Timer2 Prescale = 1:1			
  			 T2_SOURCE_INT ,								// Timer2 Clock Source = Internal
             65535);										// Timer2 Match Value 

  WriteTimer2(0);											// Reset Timer2 Count
  
  OpenCapture1(IC_IDLE_STOP &								// Disable Capture1 in IDLE Mode             
			   IC_TIMER2_SRC &								// Used Timer2 = Source Clock Capture1
               IC_INT_1CAPTURE &							// 1 Capture / Interrupt               
               IC_EVERY_FALL_EDGE); 						// Capture1 Start on Rising Edge
}

